function [h_2D, minmaxN_x, minmaxN_y] = hist2(vxy, minmaxN_x, minmaxN_y, mode, eps)
% [h_2D, MinMaxN_x, MinMaxN_y] = hist2(Data_XY, MinMaxN_x, MinMaxN_y, MODE, EPSILON)
%
% 'exact' 2D histogram - will split between adjacent bins the points 
% whose values fall in the interval (edge-EPSILON, edge+EPSILON)
% a 'fast' mode is also available; some input data protection and 
% conditioning is implemented 
%
% EPSILON   - tolerance around edge; default 1e-14
% MODE      - 'exact' (default) or 'fast'
%
% Data_XY       - 
%   if [M x 2] will split Data_X col 1, Data_Y col 2
%   if [M x 1] will assume complex data, Data_X = Re(Data_XY), Data_Y  = Im(Data_XY)
%
% MinMaxN_x, MinMaxN_y will specify the mesh as such :
%   if of length 1 : {Nbins}, will scale min-to-max 
%   if of length 2 : {min,max}, will default to Nbins=10 bins
%   if of length 3 : {min, max, Nbins}
% the adjusted mesh will be returned
%
% EXAMPLES : 
%    h2d = hist2(data); 
%       % - 'exact' histogram using a 10x10 bin mesh
%   [h2d, chk_x, chk_y] = hist2(data, 25, [0 1 10], 'fast'); 
%       % - 'fast' 25-by-10 histogram using [0 1] for y-scale
%
% to PLOT :
%   step_x = (chk_x(2)-chk_x(1))/chk_x(3); % actual bins !
%   step_y = (chk_y(2)-chk_y(1))/chk_y(3); 
%   binCenters_x = [step_x:step_x:chk_x(2)] - step_x/2;
%   binCenters_y = [step_y:step_y:chk_y(2)] - step_y/2;
%   imagesc(binCenters_y, binCenters_x, h2d); 
%
% 21.01.2008    - tudima@yahoo.com

% --- input protection/detection ---
if nargin < 5, eps = 1e-15; end;
if nargin < 4, mode = 'shared'; end;

% figure out if data transposed !
if size(vxy,1) < size(vxy,2) 
    vxy = vxy';
end;
% decompose vxy in vx vy
if size(vxy,2) >= 2 % if 3+ will keep Nx2 data
    vx = vxy(:,1); vy = vxy(:,2);
else
    vx = real(vxy); vy = imag(vxy);
end;

% 1. define minimum x, y, meshes : 
% data will not be sliced, but rather the
% mesh adjusted so that it will include all data
scale_start_x = min(min(vx));
scale_start_y = min(min(vy));
scale_end_x = max(max(vx));
scale_end_y = max(max(vy));
% default number of bins;
N_default = 10; N_y = N_default; N_x = N_default;

%  >> vector size logic : 1:N, 2:{min,max}, 3: {min, max, N}
if nargin < 3
    if nargin < 2
        minmaxN_y = [];
        minmaxN_x = [];
    else    
        minmaxN_y = minmaxN_x;
    end;    
end;

% 2.a figure out y scale
if size(minmaxN_y,2) >= 2
    scale_start_y = min(scale_start_y, minmaxN_y(1));
    scale_end_y = max(scale_end_y, minmaxN_y(2));
elseif size(minmaxN_y,2) ==1
    N_y = minmaxN_y;
end;
if size(minmaxN_y,2) ==3, N_y = max(minmaxN_y(3),0); end;
step_y = (scale_end_y-scale_start_y)/N_y; 
BinUpperEdge_y = scale_start_y + (1:N_y)*step_y;

% 2.b figure out x scale
if size(minmaxN_x,2) >= 2
    scale_start_x = min(scale_start_x, minmaxN_x(1));
    scale_end_x = max(scale_end_x, minmaxN_x(2));
elseif size(minmaxN_x,2) ==1
    N_x = minmaxN_x;
end;
if size(minmaxN_x,2) == 3, N_x = max(minmaxN_x(3),0); end;
step_x = (scale_end_x-scale_start_x)/N_x; % va dispare
BinUpperEdge_x = scale_start_x + (1:N_x)*step_x;

% 2.c recompose and export the (corrected) 3-length scale vector
minmaxN_x = [scale_start_x scale_end_x N_x];
minmaxN_y = [scale_start_y scale_end_y N_y];
% --- end input protection / conditioning ---

switch mode
    case {'shared', 'exact'}
        epsWeight = 1;           
    case {'simple', 'fast'}
        epsWeight = 0;
    otherwise
        disp('bin2D: unrecognized binning mode');
end;

% logic table of navigation parameters
% binSkip  eps_sign    bin_coeff   LowerEdge   UpperEdge       thisBin(s)
% 0         +1          1/2         th(n-1)-eps th(n-1)+eps     n-1, n
% 1         -1          1           th(n-1)+eps th(n)-eps       n
% 0         +1          1/2         th(n)-eps   th(n)+eps       n, n+1
% ...
% 0         +1          1/2         th(e-1)-eps th(e-1)+eps     e-1 (e=end)
% 1         0           1           th(e-1)+eps th(e)           end, cum ?

% --- calculate histogram ---
h_2D = zeros(N_y, N_x); 
thisBin_x = 1; 
ix_remaining_x = (1:max(size(vx)));
% init bin-navigating characteristics X, advance binSkip by binSkip :-)
epsSign_x = -1; binSkip_x = 1; binShareFactor_x = 1;
while thisBin_x <= N_x
    % check one bin width :
    threshold_x = BinUpperEdge_x(thisBin_x) + epsSign_x*eps; % -eps
    ix_this_bin_x = find(vx(ix_remaining_x)<=threshold_x); 
    ix_orig_x = ix_remaining_x(ix_this_bin_x);
    % among these ones sort y...
    thisBin_y = 1;
    ix_remaining_y = ix_orig_x; % only look at points selected by x
    % init bin-navigating characteristics Y
    epsSign_y = -1; binSkip_y = 1; binShareFactor_y = 1;
    while thisBin_y <= N_y                
        threshold_y = BinUpperEdge_y(thisBin_y)+ epsSign_y*eps; % -eps
        ix_this_bin_y = find(vy(ix_remaining_y)<=threshold_y); 
        % mark as found x,y
        ix_orig_y = ix_remaining_y(ix_this_bin_y);
        % --- update the histogram ---
        noOfElHere = length(ix_this_bin_y);
        if noOfElHere >0
            if binShareFactor_y == 1/2
                if binShareFactor_x == 1/2 % some of these coud be inits, not acc.; but easy is safe !
                    h_2D(thisBin_y,thisBin_x) = h_2D(thisBin_y,thisBin_x)+ noOfElHere/4;
                    h_2D(thisBin_y+1,thisBin_x) = h_2D(thisBin_y+1,thisBin_x)+ noOfElHere/4;
                    h_2D(thisBin_y,thisBin_x+1) = h_2D(thisBin_y,thisBin_x+1)+ noOfElHere/4;
                    h_2D(thisBin_y+1,thisBin_x+1) = h_2D(thisBin_y+1,thisBin_x+1)+ noOfElHere/4;
                else % _x 1
                    h_2D(thisBin_y,thisBin_x) = h_2D(thisBin_y,thisBin_x) +noOfElHere/2;
                    h_2D(thisBin_y+1,thisBin_x) = h_2D(thisBin_y+1,thisBin_x) +noOfElHere/2;                        
                end;
            else % _y 1
                if binShareFactor_x == 1/2
                    h_2D(thisBin_y,thisBin_x) = h_2D(thisBin_y,thisBin_x) + noOfElHere/2;
                    h_2D(thisBin_y,thisBin_x+1) = h_2D(thisBin_y,thisBin_x+1) +noOfElHere/2;
                else % 1
                    h_2D(thisBin_y,thisBin_x) = h_2D(thisBin_y,thisBin_x) + noOfElHere;                        
                end;
            end;
        end; % --- end histogram updating 
        % eliminate these y-bins
        ix_remaining_y(ix_this_bin_y) = 0;   
        ix_remaining_y = ix_remaining_y(ix_remaining_y >0);
        % sanity :
        if isempty(ix_remaining_y), break; end;
        
        % --- update navigation params Y --- advance y Bin counter ---
        epsSign_y = -epsSign_y * ~(thisBin_y == N_y-1) *epsWeight; % 0 for last bin or in 'fast' mode
        binSkip_y = ~(epsSign_y==1); 
        binShareFactor_y = (binSkip_y+1)/2;         
        thisBin_y = thisBin_y + binSkip_y;           
    end % while y
    
    % eliminate x-bins -- slow way... 
    ix_remaining_x(ix_this_bin_x) = 0;   
    ix_remaining_x = ix_remaining_x(ix_remaining_x >0);  
    % sanity :
    if isempty(ix_remaining_x), break; end;
    
    % --- update navigation params X --- advance bin counter ---
    epsSign_x = -epsSign_x * ~(thisBin_x == N_x-1) *epsWeight; % 0 for last bin or in 'fast' mode
    binSkip_x = ~(epsSign_x==1); 
    binShareFactor_x = (binSkip_x+1)/2;
    thisBin_x = thisBin_x + binSkip_x;            
end; % while x